"use client";

import { useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { Card } from "@/components/ui/Card";
import { Input } from "@/components/ui/Input";
import { Button } from "@/components/ui/Button";
import { supabaseBrowser } from "@/lib/supabase/client";

const loanTypes = [
  { code: "business", label: "Business Loan" },
  { code: "po_finance", label: "Purchase Order Finance" },
  { code: "loc", label: "Line of Credit" },
] as const;

export default function BusinessWizard() {
  const router = useRouter();
  const [step, setStep] = useState(1);
  const [loanType, setLoanType] = useState<(typeof loanTypes)[number]["code"]>("business");
  const [amount, setAmount] = useState("20000");
  const [term, setTerm] = useState("6");
  const [purpose, setPurpose] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const progress = useMemo(() => (step / 3) * 100, [step]);

  async function createDraft() {
    setLoading(true);
    setError(null);
    const supabase = supabaseBrowser();

    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      setLoading(false);
      setError("Not signed in.");
      return;
    }

    const { data: kyc } = await supabase
      .from("kyc_cases")
      .select("status")
      .eq("user_id", user.id)
      .order("created_at", { ascending: false })
      .limit(1)
      .maybeSingle();

    if (!kyc || kyc.status !== "approved") {
      setLoading(false);
      setError("KYC is required and must be approved before you can submit. Go to My Account & KYC.");
      return;
    }

    const { data: lp } = await supabase
      .from("loan_products")
      .select("id")
      .eq("type_code", loanType)
      .maybeSingle();

    if (!lp?.id) {
      setLoading(false);
      setError("Loan product not configured. Ask admin to seed loan_products.");
      return;
    }

    const { data: app, error: appErr } = await supabase
      .from("applications")
      .insert({
        loan_product_id: lp.id,
        borrower_type: "business",
        amount_requested: Number(amount),
        term_requested: Number(term),
        purpose_text: purpose,
        status: "draft"
      })
      .select("id")
      .single();

    setLoading(false);
    if (appErr) return setError(appErr.message);
    router.push(`/app/applications/${app.id}`);
  }

  return (
    <div className="space-y-6">
      <Card>
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-semibold">Business Application</h1>
            <p className="mt-1 text-sm text-slate-600">PO finance + enterprise ready structure.</p>
          </div>
          <div className="text-sm text-slate-500">{step}/3</div>
        </div>
        <div className="mt-4 h-2 w-full rounded-full bg-slate-100 overflow-hidden">
          <div className="h-full bg-thebe-blue2" style={{ width: `${progress}%` }} />
        </div>
      </Card>

      {step === 1 && (
        <Card>
          <div className="text-sm font-semibold">Step 1: Loan Type</div>
          <div className="mt-3 grid grid-cols-1 md:grid-cols-2 gap-3">
            {loanTypes.map((t) => (
              <button
                key={t.code}
                type="button"
                onClick={() => setLoanType(t.code)}
                className={`rounded-xl2 border px-4 py-3 text-left transition ${
                  loanType === t.code
                    ? "border-blue-200 bg-blue-50"
                    : "border-thebe-border bg-white hover:bg-slate-50"
                }`}
              >
                <div className="text-sm font-medium">{t.label}</div>
                <div className="text-xs text-slate-500 mt-1">Issuer underwriting applies.</div>
              </button>
            ))}
          </div>
          <div className="mt-5 flex justify-end">
            <Button onClick={() => setStep(2)}>Continue</Button>
          </div>
        </Card>
      )}

      {step === 2 && (
        <Card>
          <div className="text-sm font-semibold">Step 2: Loan Details</div>
          <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <label className="text-sm text-slate-600">Amount (BWP)</label>
              <Input value={amount} onChange={(e) => setAmount(e.target.value)} />
            </div>
            <div>
              <label className="text-sm text-slate-600">Term (months)</label>
              <Input value={term} onChange={(e) => setTerm(e.target.value)} />
            </div>
            <div className="md:col-span-2">
              <label className="text-sm text-slate-600">Purpose</label>
              <Input value={purpose} onChange={(e) => setPurpose(e.target.value)} placeholder="e.g., inventory, expansion, PO fulfilment..." />
            </div>
          </div>
          <div className="mt-5 flex justify-between">
            <Button variant="secondary" onClick={() => setStep(1)}>Back</Button>
            <Button onClick={() => setStep(3)}>Continue</Button>
          </div>
        </Card>
      )}

      {step === 3 && (
        <Card>
          <div className="text-sm font-semibold">Step 3: Submit (KYC Required)</div>
          <p className="mt-2 text-sm text-slate-600">We will route to multiple approved issuers.</p>
          {error && <div className="mt-3 text-sm text-red-600">{error}</div>}
          <div className="mt-5 flex justify-between">
            <Button variant="secondary" onClick={() => setStep(2)}>Back</Button>
            <Button onClick={createDraft} disabled={loading}>
              {loading ? "Creating..." : "Create application"}
            </Button>
          </div>
        </Card>
      )}
    </div>
  );
}
