import { NextResponse, type NextRequest } from "next/server";
import { createServerClient } from "@supabase/ssr";

const PROTECTED = [
  { prefix: "/app", roles: ["borrower_personal", "borrower_business", "admin"] },
  { prefix: "/issuer", roles: ["issuer_user", "admin"] },
  { prefix: "/admin", roles: ["admin"] },
];

function isProtected(pathname: string) {
  return PROTECTED.find((p) => pathname === p.prefix || pathname.startsWith(p.prefix + "/"));
}

export async function middleware(req: NextRequest) {
  const res = NextResponse.next();

  const protectedRule = isProtected(req.nextUrl.pathname);
  if (!protectedRule) return res;

  const supabase = createServerClient(
    process.env.NEXT_PUBLIC_SUPABASE_URL!,
    process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY!,
    {
      cookies: {
        getAll() {
          return req.cookies.getAll();
        },
        setAll(cookiesToSet) {
          cookiesToSet.forEach(({ name, value, options }) =>
            res.cookies.set(name, value, options)
          );
        },
      },
    }
  );

  const { data: { user } } = await supabase.auth.getUser();
  if (!user) {
    const url = req.nextUrl.clone();
    url.pathname = "/login";
    url.searchParams.set("next", req.nextUrl.pathname);
    return NextResponse.redirect(url);
  }

  // fetch profile role
  const { data: profile } = await supabase
    .from("profiles")
    .select("role")
    .eq("user_id", user.id)
    .maybeSingle();

  const role = profile?.role ?? "borrower_personal";
  if (!protectedRule.roles.includes(role)) {
    const url = req.nextUrl.clone();
    url.pathname = "/app";
    return NextResponse.redirect(url);
  }

  return res;
}

export const config = {
  matcher: ["/app/:path*", "/issuer/:path*", "/admin/:path*"],
};
